﻿using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Globalization;
using SoftPi.Tariscope.Common;
using System.Data;

namespace SoftPi.Tariscope.WebAdministration.Scripts
{
    /// <summary>
    /// Обработка расшифровки от оператора МТС (Vodafone). (2021.09) Украина.
    /// </summary>
    public class Ua_Vodafone_2021 : IScript
    {
        /// <summary>
        /// Не фильтровать записи по услугам, которые содержат итоговые значения. 
        /// Если True, часть сумм будет дублироваться в итоговых строках.
        /// </summary>
        public bool ProcessAllServices = false;

        /// <summary>
        /// Сохранять детали по вызовам в базу вызовов (Calls). 
        /// Не стоит использовать одновременно с ProcessCallServices.
        /// </summary>
        public bool ProcessCallDetails = true;
        
        /// <summary>
        /// Сохранять в перечне услуг услуги, которые относятся к вызовам/смс.
        /// Не стоит использовать одновременно с ProcessCallDetails.
        /// </summary>
        public bool ProcessCallServices = false;
        
        /// <summary>
        /// Обрабатывать услуги.
        /// </summary>
        public bool ProcessServices = true;
        
        /// <summary>
        /// Сохранять услуги в базу вызовов. Если выключено, то услуги будут сохраняться в таблицу AccountHistory.
        /// </summary>
        public bool ProcessServicesAsCalls = false;
        
        /// <summary>
        /// Если флаг установлен, то для вызовов, записываемых в базу, стоимость будет 
        /// </summary>
        public bool TariffCallsCost = false;

        private const string ABONENT_ID_PARAMETER = "AbonentId";
        private const string CURRENT_DN_PARAMETER = "CurrentDn";
        private const string DN_ID_PARAMETER = "DNID";
        private const string PBX_ID_PARAMETER = "PBXID";

        private const string SERVICE_DATE_PARAMETER = "ServiceDate";
        
        private const string SERVICE_NAME = "MTS Services";

        private int abonentID;
        private readonly List<string> callServices = new List<string>();
        private string currentDN;

        private IScriptDatabaseHelper databaseHelper;
        private int dnId;
        private IScriptHost host;
        private readonly List<string> ignoredServices = new List<string>();
        private int pbxId;
        private int rootDepartment;
        private DateTime serviceDate;

        // Current parse state variables
        private int serviceId;

        private CultureInfo culture = CultureInfo.GetCultureInfoByIetfLanguageTag("uk-ua");

        public void Init(IScriptHost host, IScriptDatabaseHelper dbHelper)
        {
            this.host = host;
            this.databaseHelper = dbHelper;
            ignoredServices.Add("ЗАГАЛОМ ЗА КОНТРАКТОМ");
            ignoredServices.Add("ПОСЛУГИ_ НАДАНІ ЗА МЕЖАМИ ПАКЕТА");
            ignoredServices.Add("КОНТЕНТ-ПОСЛУГИ");
            ignoredServices.Add("НАДАНІ КОНТЕНТ-ПОСЛУГИ");
            ignoredServices.Add("ЗАМОВЛЕНІ КОНТЕНТ-ПОСЛУГИ");
            ignoredServices.Add("ЗАМОВЛЕНІ ДОДАТКОВІ ПОСЛУГИ ЗА МЕЖАМИ ПАКЕТА");
            ignoredServices.Add("ПОСЛУГИ МІЖНАРОДНОГО РОУМІНГУ");
            ignoredServices.Add("ЗНИЖКИ");
            //ignoredServices.Add("СПЕЦІАЛЬНІ ПОСЛУГИ:");

            callServices.Add("Вихідні дзвінки_ SMS_ передача даних");
            callServices.Add("Вхідні дзвінки");
        }

        public void Main(object fileName)
        {
            ReadInitialVariables();

            var fileContent = File.ReadAllLines(fileName as string, Encoding.Default);
            var lineIndex = 0;

            var forLine = string.Empty;

            try
            {
                foreach (var line in fileContent)
                {
                    forLine = line;

                    lineIndex += 1;

                    if (host.IsCancelling)
                        return;

                    if (string.IsNullOrEmpty(line))
                        continue;

                    var cleanLine = PrepareCsvRow(line);

                    var columns = CsvHelper.SplitCSV(cleanLine);

                    //Logger.Info($"Line: {line}");
                    //Logger.Info($"Clean line: {cleanLine}");
                    //Logger.Info($"Columns: {string.Join('|', columns)}");

                    try
                    {
                        var callTypeColumnText = columns[2].Trim();

                        if (columns[0].StartsWith("Контракт №"))
                            UpdateContractInfo(columns, rootDepartment);
                        else if (callTypeColumnText == "Вхідні дзвінки")
                            SaveIncomingCall(columns);
                        else if (callTypeColumnText == "Вихідні дзвінки")
                            SaveOutgoingCall(columns);
                        else if (callTypeColumnText == "Вихідні повідом.")
                            SaveOutgoingSms(columns);
                        else if (callTypeColumnText == "Вхідні повідом.")
                            SaveIncomingSms(columns);
                        else if (callTypeColumnText == "GPRS/CDMA з'єд.") { }
                        else if (columns[0].StartsWith("Період:"))
                        {
                            serviceDate = DateTime.Parse(columns[0].Substring(8, 10).Trim());
                            DoCleanup(serviceId, serviceDate);
                        }
                        else if (columns[0].StartsWith("Розрахунковий період:"))
                        {
                            serviceDate = DateTime.Parse(columns[0].Substring(22, 10).Trim());
                            DoCleanup(serviceId, serviceDate);
                        }
                        else if (ProcessServices 
                              && !string.IsNullOrEmpty(currentDN)
                              && !columns[0].StartsWith("Тарифний Пакет"))
                            SaveService(columns); 
                    }
                    catch (Exception ex) 
                    {
                        Logger.Info($"Import exception: {ex.Message} \n {ex.StackTrace}");
                    }
                } 
            }
            catch (Exception ex)
            {
                throw new Exception(string.Format("Exception in line:{0} {1}", lineIndex, forLine), ex);
            }

            SaveInitialVariables();
        }

        private bool IsIgnoredService(string serviceName)
        {
            if (ProcessAllServices)
                return false;

            foreach (var itm in ignoredServices)
                if (serviceName.StartsWith(itm, StringComparison.CurrentCultureIgnoreCase))
                    return true;

            if (!ProcessCallServices)
                foreach (var itm in callServices)
                    if (serviceName.StartsWith(itm, StringComparison.CurrentCultureIgnoreCase))
                        return true;
            
            return false;
        }

        /// <summary>
        /// Get service id or create new
        /// </summary>
        /// <param name="ServiceName"></param>
        /// <returns></returns>
        /// <remarks></remarks>
        private int GetServiceId(string serviceName)
        {
            using (var cn = databaseHelper.GetConnection())
            {
                var cmd = databaseHelper.GetQuery("SELECT ID FROM Services WHERE Name=@name", cn);
                databaseHelper.AddParameter("@name", serviceName, cmd);

                using (IDataReader rs = cmd.ExecuteReader())
                {
                    if (rs.Read())
                        return rs.GetInt32(0);
                }

                //no service found, create one
                cmd = databaseHelper.GetCommand("ab_mc_serviceadd", cn, true);
                cmd.Parameters["@name"].Value = serviceName;
                cmd.Parameters["@period"].Value = 0;
                cmd.ExecuteNonQuery();
                return (int)cmd.Parameters[0].Value;
            }
        }
        
        /// <summary>
        /// Find root department of selected PBX
        /// </summary>
        /// <returns></returns>
        /// <remarks></remarks>
        private int GetRootDepartment()
        {
            int nodeId = new();

            using (var cn = databaseHelper.GetConnection())
            {
                var cmd = databaseHelper.GetCommand("ab_mc_pbxdetails", cn, true);
                cmd.Parameters["@id"].Value = host.GetProperty("PBXID");

                using (IDataReader rs = cmd.ExecuteReader())
                {
                    if (rs.Read())
                        nodeId = (int)rs["NodeId"];
                }

                cmd = databaseHelper.GetCommand("ab_mc_departmentroot", cn, true);
                cmd.Parameters["@nodeid"].Value = nodeId;

                using (IDataReader rs = cmd.ExecuteReader())
                {
                    if (rs.Read())
                        return (int)rs["ID"];
                    else
                        return 0;
                }
            }
        }

        /// <summary>
        /// Convert string duration representation of MTS CSV format to TimeSpan
        /// </summary>
        /// <param name="durationString">string representation of time span. <example>0:25</example></param>
        /// <returns>Duration as Timespan value</returns>
        private TimeSpan GetDuration(string durationString)
        {
            try
            {
                durationString = durationString.Trim();
                TimeSpan duration;
                var delimitersCount = GetOccurencesCount(":", durationString);
                string[] parts;
                switch (delimitersCount)
                {
                    case 2:
                        parts = durationString.Split(':');
                        return new TimeSpan(int.Parse(parts[0]), int.Parse(parts[1]), int.Parse(parts[2]));
                    case 1:
                        parts = durationString.Split(':');
                        return new TimeSpan(0, int.Parse(parts[0]), int.Parse(parts[1]));
                    case 0:
                        return new TimeSpan(0, 0, int.Parse(durationString));
                    default:
                        throw new Exception();
                }
            }
            catch 
            {
                return TimeSpan.Zero;
            }
        }

        private int GetDNIDByDN(int abonentId, string dn)
        {
            using (var cn = databaseHelper.GetConnection())
            {
                var cmd = databaseHelper.GetCommand("ab_mc_dnidbydn", cn);
                cmd.Parameters["@abonentid"].Value = abonentId;
                cmd.Parameters["@dn"].Value = dn;
                cmd.ExecuteNonQuery();
                return (int)cmd.Parameters[0].Value;
            }
        }

        private int GetAbonentByDn(string dn, int rootDepartment)
        {
            using (var cn = databaseHelper.GetConnection())
            {
                var cmd = databaseHelper.GetCommand("ab_mc_abonentsearch2", cn);
                databaseHelper.SetParameter("@dn", dn, cmd);

                using (IDataReader rs = cmd.ExecuteReader())
                {
                    if (rs.Read())
                    {
                        return (int)rs["ID"];
                    }
                    else
                    {
                        rs.Close();
                        cmd = databaseHelper.GetCommand("ab_mc_abonentadd", cn);
                        databaseHelper.SetParameter("@lastname", dn, cmd);
                        databaseHelper.SetParameter("@departmentid", rootDepartment, cmd);
                        cmd.ExecuteNonQuery();

                        var abonentID = (int)databaseHelper.GetParameter(0, cmd);
                        cmd = databaseHelper.GetCommand("ab_mc_dnadd", cn);
                        cmd.Parameters["@abonentid"].Value = abonentID;
                        cmd.Parameters["@dn"].Value = dn;
                        cmd.ExecuteNonQuery();
                        return abonentID;
                    }
                }
            }
        }

        private int GetOccurencesCount(string needle, string haystack)
        {
            return (haystack.Length - haystack.Replace(needle, string.Empty).Length) / needle.Length;
        }

        private void SaveInitialVariables()
        {
            host.SetProperty(SERVICE_DATE_PARAMETER, serviceDate);
            host.SetProperty(ABONENT_ID_PARAMETER,   abonentID);
            host.SetProperty(DN_ID_PARAMETER,        dnId);
            host.SetProperty(CURRENT_DN_PARAMETER,   currentDN);
        }

        private void ReadInitialVariables()
        {
            serviceId = GetServiceId(SERVICE_NAME);
            rootDepartment = GetRootDepartment();

            serviceDate = (DateTime)host.GetProperty(SERVICE_DATE_PARAMETER, DateTime.Now);
            abonentID = (int)host.GetProperty(ABONENT_ID_PARAMETER, 0);

            dnId = (int)host.GetProperty(DN_ID_PARAMETER, 0);
            currentDN = host.GetProperty(CURRENT_DN_PARAMETER, string.Empty).ToString();
            pbxId = (int)host.GetProperty(PBX_ID_PARAMETER);
        }

        private void SaveService(string[] columns)
        {
            var serviceDescription = columns[0];
            var amount = columns[3];
            decimal testAmount;

            if (amount == string.Empty && decimal.TryParse(columns[1], NumberStyles.Number, CultureInfo.InvariantCulture, out testAmount))
                amount = columns[1];

            var amount2 = decimal.Parse(amount, NumberStyles.Number, CultureInfo.InvariantCulture);

            serviceDescription = serviceDescription.Replace("Вартість послуги", "").Replace(":", "").Trim();

            if (serviceDescription.Length > 0
             && !IsIgnoredService(serviceDescription)
             && amount.Length > 0 && amount2 != 0)
            {
                if (ProcessServicesAsCalls)
                {
                    var callItem = new CallModel
                    {
                        RecordType = RecordTypeEnum.Realtime,
                        CallDuration = 0,
                        CallDateTime = serviceDate,
                        Originator = currentDN,
                        Terminator = "SERVICE",
                        DialNumber = serviceDescription,
                        CLID = string.Empty,
                        Cost = amount2
                    };
                    host.AddCall(callItem);
                }
                else
                {
                    var newService = new ServiceModel
                    {
                        ServiceId = serviceId,
                        AbonentId = abonentID,
                        DNId = dnId,
                        ServiceDate = serviceDate,
                        Cost = amount2,
                        ServiceDescription = serviceDescription
                    };
                    host.AddService(newService);
                }
            }
        }

        private void DoCleanup(int serviceId, DateTime serviceDate)
        {
            using (var cn = databaseHelper.GetConnection())
            {
                var cmd = databaseHelper.GetQuery("DELETE FROM AccountHistory WHERE RecDate=@servicedate AND ServiceID IN (SELECT ID FROM AbonentServices WHERE ServiceID=@serviceid)", cn);
                databaseHelper.AddParameter("@servicedate", serviceDate, cmd);
                databaseHelper.AddParameter("@serviceid", serviceId, cmd);
                cmd.CommandType = CommandType.Text;
                cmd.ExecuteNonQuery();
            }
        }

        private void SaveIncomingSms(string[] columns)
        {
            if (!ProcessCallDetails)
                return;

            var numberB = columns[4].Trim();
            var callDateTime = DateTime.Parse(columns[5], culture).Add(TimeSpan.Parse(columns[6]));
            var duration = GetDuration(columns[7]);
            var serviceCost = decimal.Parse(columns[9], NumberStyles.Number, CultureInfo.InvariantCulture);
            var callItem = new CallModel();

            if (TariffCallsCost)
                callItem.RecordType = RecordTypeEnum.Regular;
            else
                callItem.RecordType = RecordTypeEnum.Imported;

            callItem.CallDuration = (int)duration.TotalSeconds;
            callItem.CallDateTime = callDateTime;
            callItem.Originator = "L000000";
            callItem.Terminator = currentDN;
            callItem.DialNumber = "";
            callItem.CLID = numberB;
            callItem.Cost = serviceCost;
            callItem.CallType = 13;
            host.AddCall(callItem);
        }

        private void SaveOutgoingSms(string[] columns)
        {
            if (!ProcessCallDetails)
                return;

            var numberB = columns[4].Trim();
            var callDateTime = DateTime.Parse(columns[5], culture).Add(TimeSpan.Parse(columns[6]));
            var duration = GetDuration(columns[7]);
            var serviceCost = decimal.Parse(columns[9], NumberStyles.Number, CultureInfo.InvariantCulture);
            var callItem = new CallModel();

            if (TariffCallsCost)
                callItem.RecordType = RecordTypeEnum.Regular;
            else
                callItem.RecordType = RecordTypeEnum.Imported;

            callItem.CallDuration = (int)duration.TotalSeconds;
            callItem.CallDateTime = callDateTime;
            callItem.Originator = currentDN;
            callItem.Terminator = "L000000";
            callItem.DialNumber = numberB;
            callItem.CLID = "";
            callItem.Cost = serviceCost;
            callItem.CallType = 13;
            host.AddCall(callItem);
        }

        private void SaveIncomingCall(string[] columns)
        {
            if (!ProcessCallDetails) 
                return;

            var numberB = columns[4].Trim();
            var callDateTime = DateTime.Parse(columns[5], culture).Add(TimeSpan.Parse(columns[6]));
            var duration = GetDuration(columns[7]);
            var serviceCost = decimal.Parse(columns[9], NumberStyles.Number, CultureInfo.InvariantCulture);

            var callItem = new CallModel();

            if (TariffCallsCost)
                callItem.RecordType = RecordTypeEnum.Regular;
            else
                callItem.RecordType = RecordTypeEnum.Imported;

            callItem.CallDuration = (int)duration.TotalSeconds;
            callItem.CallDateTime = callDateTime;
            callItem.Originator = "L000000";
            callItem.Terminator = currentDN;
            callItem.DialNumber = "";
            callItem.CLID = numberB;
            callItem.Cost = serviceCost;
            host.AddCall(callItem);
        }

        private void SaveOutgoingCall(string[] columns)
        {
            if (!ProcessCallDetails)
                return;

            var numberB = columns[4].Trim();
            var callDateTime = DateTime.Parse(columns[5], culture).Add(TimeSpan.Parse(columns[6]));
            var duration = GetDuration(columns[7]);
            var serviceCost = decimal.Parse(columns[9], NumberStyles.Number, CultureInfo.InvariantCulture);
            var callItem = new CallModel();

            if (TariffCallsCost)
                callItem.RecordType = RecordTypeEnum.Regular;
            else
                callItem.RecordType = RecordTypeEnum.Imported;

            callItem.CallDuration = (int)duration.TotalSeconds;
            callItem.CallDateTime = callDateTime;
            callItem.Originator = currentDN;
            callItem.Terminator = "L000000";
            callItem.DialNumber = numberB;
            callItem.CLID = "";
            callItem.Cost = serviceCost;
            host.AddCall(callItem);
        }

        private void UpdateContractInfo(string[] columns, int rootDepartment)
        {
            currentDN = string.Empty;

            var pos = columns[0].IndexOf("Номер телефону:", StringComparison.Ordinal);

            if (pos > 0)
                currentDN = columns[0][(pos + 15)..].Trim();
            else
                currentDN = columns[0][10..].Trim();

            if (currentDN.Length > 0)
            {
                abonentID = GetAbonentByDn(currentDN, rootDepartment);
                dnId = GetDNIDByDN(abonentID, currentDN);
            }
        }

        private string PrepareCsvRow(string line)
        {
            var cleanLine = line.Replace(";", string.Empty).Replace(@"""""", ""); //.Replace(", ", "_ ")
            //Now we need to fix incorrect CSV format, then CSV cell contains "," but not quoted in any way.

            var sb = new StringBuilder();

            for (var i = 0; i < cleanLine.Length; i++)
            {
                if (cleanLine[i] == ',')
                {
                    if (i >= cleanLine.Length - 1) { }

                    else if (cleanLine[i + 1] != ' ')
                    {
                        sb.Append(cleanLine[i..]);
                        break;
                    }
                    else
                    {
                        sb.Append('_');
                        continue;
                    }
                }
                sb.Append(cleanLine[i]);
            }

            cleanLine = sb.ToString();

            if (cleanLine.StartsWith(@"""") && cleanLine.EndsWith(@""""))
            {
                cleanLine = cleanLine[1..];
                cleanLine = cleanLine[0..^1];
            }

            cleanLine += ",,,,,";
            return cleanLine;
        }
    }
}
